/**
 * \file: pfcfg_filebackend.h
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * File backend for Platform Configuration (pfcfg).
 * Backend for reading configuration files out of the filesystem and
 * parsing them.
 *
 * \component: pfcfg
 *
 * \author: M. Methner
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \history
 * 0.1 M. Methner Initial version
 *
 ***********************************************************************/

#include "pfcfg_interface.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


struct PFCFG_Handle_t
{
    FILE * p_fid;
};



LOCAL U32 min(U32 a, U32 b)
{
    U32 retval;
    if (a < b)
    {
        retval = a;
    }
    else
    {
        retval = b;
    }
    return retval;
}



PFCFG_CHAR *
PFCFG_skip_whitespaces(PFCFG_CHAR * p_str, U32 length)
{
    PFCFG_CHAR * p_offset = p_str;

    while (((U32)(p_offset - p_str) < length) &&
           (*p_offset == ' ' || *p_offset == '\t') && *p_offset != '\0')
    {
        p_offset++;
    }

    return p_offset;
}

U32
PFCFG_get_keylength(PFCFG_CHAR * p_str)
{
    PFCFG_CHAR * p_offset = p_str;

    while ((*p_offset != ' ' && *p_offset != '\t') && *p_offset != '\0')
    {
        p_offset++;
    }

    return (U32)(p_offset-p_str);
}


PFCFG_Handle_t *
PFCFG_opn_cnf(const PFCFG_CHAR * p_name, PFCFG_MODE mode)
{
    PFCFG_CHAR dir_name[] = "/etc/pfcfg/";
    PFCFG_CHAR file_ext[] = ".cfg";
    PFCFG_CHAR * p_filename;
    U32 filename_length;
    FILE * p_fid = NULL;
    PFCFG_Handle_t * handle = NULL;

    filename_length = strlen(dir_name) + strlen(p_name) + strlen(file_ext) + 1;
    p_filename = malloc(filename_length);

    if (p_filename)
    {
        snprintf(p_filename, filename_length, "%s%s%s", dir_name, p_name, file_ext);
        if (mode == PFCFG_READ)
        {
            p_fid = fopen(p_filename, "rt");
        }
        else
        {
            /* not implemented, returns NULL */
        }
        free(p_filename);
    }

    if(p_fid)
    {
        handle=malloc(sizeof(PFCFG_Handle_t));
        if(handle)
        {
            handle->p_fid = p_fid;
        }
        else
        {
            fclose(p_fid);
        }
    }

    return handle;
}



PFCFG_ERR
PFCFG_close_cnf(PFCFG_Handle_t * p_handle)
{
    if(p_handle)
    {
        if(p_handle->p_fid)
        {
            fclose(p_handle->p_fid);
        }
        free(p_handle);
    }
    return PFCFG_NO_ERROR;
}



PFCFG_ERR
PFCFG_get_cfs(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key,
        PFCFG_CHAR * p_str, U32 length)
{
    PFCFG_ERR er = PFCFG_KEY_NOT_FOUND;
    PFCFG_CHAR cfg_line[MAX_CFG_LINE_LENGTH];
    U32 key_length = strlen(p_key);
    BOOL match = FALSE;

    /* set seek pointer to beginning of the file */
    fseek(p_handle->p_fid, 0, SEEK_SET);

    while (fgets(cfg_line, MAX_CFG_LINE_LENGTH, p_handle->p_fid) && !match)
    {
        PFCFG_CHAR * p_current_key = NULL;
        cfg_line[MAX_CFG_LINE_LENGTH-1]='\0';

        if (cfg_line[0] == '#')
        {
            continue;
        }

        p_current_key = PFCFG_skip_whitespaces(cfg_line, MAX_CFG_LINE_LENGTH);

        if ((U32) (MAX_CFG_LINE_LENGTH - (p_current_key - cfg_line))
                >= key_length)
        {
            U32 current_keylength = PFCFG_get_keylength(p_current_key);
            if (current_keylength == key_length &&
                0 == strncmp(p_current_key, p_key, key_length))
            {
                PFCFG_CHAR * p_data = p_current_key + key_length;
                match = TRUE;

                p_data = PFCFG_skip_whitespaces(p_data, MAX_CFG_LINE_LENGTH
                        - (p_data - cfg_line));

                if ((p_data - cfg_line) < MAX_CFG_LINE_LENGTH)
                {
                    S32 i;
                    U32 data_length = min(length,
                            (U32)(MAX_CFG_LINE_LENGTH - (p_data - cfg_line)));
                    strncpy(p_str, p_data, data_length);
                    if (data_length == length)
                    {
                        p_str[data_length - 1] = '\0';
                    }
                    else
                    {
                        p_str[data_length] = '\0';
                    }

                    /* remove line feed and carriage return */
                    data_length = strlen(p_str);
                    for(i=data_length-1;i>=0;i--)
                    {
                        if(isspace(p_str[i]))
                        {
                            p_str[i]='\0';
                        }
                        else
                        {
                            break;
                        }
                    }
                }
                else
                {
                    if (length > 0)
                    {
                        p_str[0] = '\0';
                    }
                }
                er = PFCFG_NO_ERROR;
            }
        }
    }
    return er;
}



PFCFG_ERR
PFCFG_get_cfn(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, S32 * p_values, U32 length)
{
    PFCFG_ERR er;
    PFCFG_CHAR cfg_line[MAX_CFG_LINE_LENGTH];
    er = PFCFG_get_cfs(p_handle, p_key, cfg_line, MAX_CFG_LINE_LENGTH);
    if (PFCFG_NO_ERROR == er)
    {
        U32 i = 0;
        PFCFG_CHAR * p_split_str = strtok(cfg_line, ", ");
        while (p_split_str)
        {
            S32 value;
            if (sscanf(p_split_str, "%i", &value))
            {
                if (i < length)
                {
                    p_values[i] = value;
                }
                i++;
            }

            p_split_str = strtok(NULL, ", ");

        }
        er = (PFCFG_ERR) i;
    }

    return er;
}



PFCFG_ERR
PFCFG_set_cfs(PFCFG_Handle_t * p_handle, PFCFG_CHAR const * p_key, const PFCFG_CHAR * p_str, U32 length)
{
    p_handle = p_handle;
    p_key = p_key;
    p_str = p_str;
    length = length;
    return PFCFG_NOT_IMPLEMENTED;
}




PFCFG_ERR
PFCFG_set_cfn(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, const S32 * p_values, U32 length)
{
    p_handle = p_handle;
    p_key = p_key;
    p_values = p_values;
    length = length;
    return PFCFG_NOT_IMPLEMENTED;
}



PFCFG_ERR
PFCFG_set_cfb(PFCFG_Handle_t * p_handle, const PFCFG_CHAR * p_key, const VP * p_data, U32 length)
{
    p_handle = p_handle;
    p_key = p_key;
    p_data = p_data;
    length = length;
    return PFCFG_NOT_IMPLEMENTED;
}

